<?php

namespace m360sync;

use m360sync\ApiFunc;

class FunctionsClass
{

    static function init()
    {
        add_action('init', [__CLASS__, 'registerPostType']);
        add_action('admin_menu', [__CLASS__, 'registerMenuPage']);
        add_action('rest_api_init', [__CLASS__, 'registerApiRoute']);
        add_filter('rest_url_prefix', [__CLASS__, 'changePrefixRoute']);
        add_action('admin_enqueue_scripts', [__CLASS__, 'pluginEnqueueStyles']);
        add_action('add_meta_boxes', [__CLASS__, 'stAddCustomBox']);
        add_action('save_post', [__CLASS__, 'projectUUIDSave']);
        add_action('wp_ajax_updateByUuid', [__CLASS__, 'updateByUuid']);
        add_action('wp_ajax_nopriv_updateByUuid', [__CLASS__, 'updateByUuid']);
        add_action('wp_ajax_fetchAllProjects', [__CLASS__, 'fetchAllProjects']);
        add_action('wp_ajax_nopriv_fetchAllProjects', [__CLASS__, 'fetchAllProjects']);
    }

    static function pluginEnqueueStyles()
    {
        wp_enqueue_style('m360-sync-admin-style', M360_SYNC_PLUGIN_URL . '/assets/css/admin-style.css');
        wp_enqueue_script('m360-sync-admin-script', M360_SYNC_PLUGIN_URL . '/assets/js/admin-script.js', array('jquery'), time(), true);
        wp_localize_script('m360-sync-admin-script', 'filters_ajax',
            [
                'url' => admin_url('admin-ajax.php'),
            ]
        );
    }


    static function registerPostType()
    {
        if (post_type_exists('prosjekt')){
            return false;
        }

        $labels = array(
            'name' => __('Projects', 'm360-sync'),
            'singular_name' => __('Project', 'm360-sync'),
            'add_new_item' => __('Add New Project', 'm360-sync'),
            'view_item' => __('View Project', 'm360-sync'),
            'search_items' => __('Search Projects', 'm360-sync'),
            'not_found' => __('No projects found', 'm360-sync'),
            'not_found_in_trash' => __('No projects found in Trash', 'm360-sync'),
            'menu_name' => __('Projects', 'm360-sync')
        );

        $args = array(
            'labels' => $labels,
            'description' => '',
            'public' => true,
            'menu_position' => 20,
            'menu_icon' => 'dashicons-hammer',
            'hierarchical' => false,
            'show_in_rest' => true,
            'supports' => ['title', 'thumbnail', 'editor', 'author'],
            'has_archive' => false,
        );
        register_post_type('prosjekt', $args);
    }

    static function registerApiRoute()
    {
        register_rest_route('v1', '/push-project/', array(
            'methods' => 'POST',
            'callback' => [__CLASS__, 'apiRouteCallback'],
            'show_in_index' => false,
            'permission_callback' => '__return_true',

        ));
    }

    static function apiRouteCallback($request)
    {
        $uuid = $request->get_param('uuid');
        $clientId = get_option('m360-sync-client-id');
        $clientSecret = get_option('m360-sync-client-secret');
        if (!$clientId || !$clientSecret) {
            $response = [
                'error' => 'In order to get Api Route please fill in Client Id and Secret Key'
            ];
            http_response_code(400);
        } elseif (!$uuid && empty($uuid)) {
            $response = [
                'error' => 'uuid not specified'
            ];
            http_response_code(400);
        } else {
            $getResponse = ApiFunc::instance()->getProjectWithResources($uuid);
            if (empty($getResponse)) {
                $response = [
                    'error' => 'Wrong uuid'
                ];
                http_response_code(400);
            } else {
                $status = self::insertProject($getResponse, $uuid);

                $response = [
                    'status' => 'Success',
                    'text' => $status
                ];
                http_response_code(200);
            }
        }
        $jsonResponse = $response;
        header('Content-Type: application/json');
        return $jsonResponse;

    }

    static function registerMenuPage()
    {
        add_menu_page(
            __('M360 Sync', 'm360-sync'),
            __('M360 Sync', 'm360-sync'),
            'manage_options',
            'm360-sync-plugin-settings',
            [__CLASS__, 'm360SettingsPage'],
            'dashicons-admin-generic'
        );
    }

    static function m360SettingsPage()
    {
        if (isset($_POST['m360-sync-settings-submit'])) {
            update_option('m360-sync-client-id', $_POST['m360-sync-client-id']);
            update_option('m360-sync-client-secret', $_POST['m360-sync-client-secret']);
            update_option('m360-sync-set-featured-image', !!$_POST['m360-sync-set-featured-image']);
            echo '<div class="notice notice-success sf-succes"><p>' . __('Settings saved successfully.', 'm360-sync') . '</p></div>';
        }
        echo '<div class="notice notice-success  sf-succes ajax-upadate"><p>' . __('Projects updated successfully.', 'm360-sync') . '</p></div>';
        $clientId = get_option('m360-sync-client-id');
        $clientSecret = get_option('m360-sync-client-secret');
        $setFeaturedImage = get_option('m360-sync-set-featured-image');
        ob_start();
        ?>
        <div class="m360-syncsettings-page">
            <h1><?php _e('M360 Settings', 'm360-sync') ?></h1>
            <div class="m360-syncapi-route-block">
                <?php if ($clientId && $clientSecret) { ?>
                    <p>
                        <b><?php _e('Webhook URL for updating a single project: ', 'm360-sync') ?></b><?php echo get_rest_url(null, 'v1/push-project') ?>
                    </p>
                <?php } else { ?>
                    <p>
                        <b><?php _e('In order to get Api Route please fill in Client Id and Secret Key', 'm360-sync'); ?></b>
                    </p>
                <?php } ?>
            </div>
            <form method="post" action="">
                <?php wp_nonce_field('m360-sync-settings', 'm360-sync-settings-nonce'); ?>
                <div class="m360-syncinput-settings">
                    <label for="m360-sync-client-id"><?php _e('Client Id:', 'm360-sync') ?></label>
                    <input type="text" name="m360-sync-client-id" id="m360-sync-client-id"
                           value="<?php echo esc_attr($clientId); ?>"/>
                </div>
                <div class="m360-syncinput-settings">
                    <label for="m360-sync-client-secret"><?php _e('Secret Key:', 'm360-sync') ?></label>
                    <input type="password" name="m360-sync-client-secret" id="m360-sync-client-secret"
                           value="<?php echo esc_attr($clientSecret); ?>"/>
                </div>
                <div class="m360-syncinput-settings">
                    <label for="m360-sync-set-featured-image"><?php _e('Update featured images:', 'm360-sync') ?></label>
                    <input type="checkbox" name="m360-sync-set-featured-image" id="m360-sync-set-featured-image" <?= $setFeaturedImage ? 'checked' : '' ?> />
                </div>
                <div class="m360-syncsettings-bottom">
                    <input type="submit" name="m360-sync-settings-submit" class="button button-primary"
                           value="<?php _e('Save Settings', 'm360-sync') ?>"/>

                    <?php if ($clientId && $clientSecret): ?>
                        <h2 style="margin-top: 2em;">Actions</h2>
                        <div class="m360-syncbutton-block">
                            <button class="fetch-existing-projects-btn" type="button"><?php _e('Update WP data for existing projects only', 'm360-sync'); ?></button>
                        </div>
                        <div class="m360-syncbutton-block">
                            <button class="fetch-all-projects-btn" type="button"><?php _e('Update WP data for all projects in API', 'm360-sync'); ?></button>
                        </div>
                    <?php endif; ?>
                </div>
            </form>
        </div>
        <?php
        $html = ob_get_contents();
        ob_clean();
        echo $html;
    }

    static function changePrefixRoute($prefix)
    {
        return 'api';
    }

    static function insertProject($respons, $uuid = null)
    {
        global $wpdb;
        $postTable = $wpdb->prefix . 'posts';
        $postMetaPost = $wpdb->prefix . 'postmeta';
        $postExist = $wpdb->get_results("SELECT pm.`meta_value`, p.ID FROM $postMetaPost as pm
        LEFT JOIN $postTable as p ON p.ID = pm.post_id
        WHERE p.post_type = 'prosjekt' AND pm.meta_key = 'project-uuid'", ARRAY_A);

        $postExists = false;
        $updatePostId = 0;

        foreach ($postExist as $postData){
            $uuidPost = $postData['meta_value'];
            if ($uuid == $uuidPost) {
                $postExists = true;
                $updatePostId = $postData['ID'];
            }
        }

        if (!$postExists) {
            $my_post = array(
                'post_title' => wp_strip_all_tags($respons->name),
                'post_content' => '',
                'post_status' => 'draft',
                'post_author' => 1,
                'post_type' => 'prosjekt',
            );
            $postId = wp_insert_post($my_post);
            update_post_meta($postId, 'project-uuid', $respons->uuid);
            update_post_meta($postId, 'st_project_data', $respons);
            self::setFeaturedImage($postId, $getResponse);
            return 'A new project has been created';
        } else {
            update_post_meta($updatePostId, 'st_project_data', $respons);
            self::setFeaturedImage($postId, $getResponse);
            return 'The project has been successfully updated';
        }
    }

    static function stAddCustomBox()
    {
        $screens = ['prosjekt'];
        foreach ($screens as $screen) {
            add_meta_box(
                'project_uuid',
                'Project UUID',
                [__CLASS__, 'projectUUID'],
                $screen
            );
        }
    }

    static function projectUUID($post)
    {
        $uuid = get_post_meta($post->ID, 'project-uuid', 'true');
        $projectData = get_post_meta($post->ID, 'st_project_data', 'true');
        ?>
        <div class="project-admin-uuid">
            <input name="project-uuid" type="text" max="12" value="<?php echo $uuid ? $uuid : '' ?>">
        </div>
        <div class="project-object">
            <div>
                <b>Project data</b>
            </div>
            <pre>
<?php echo wp_json_encode($projectData, JSON_PRETTY_PRINT); ?>
            </pre>
        </div>
        <?php
    }

    static function projectUUIDSave($post_id)
    {
        if (!isset($_POST['project-uuid'])) {
            return false;
        }
        if (!$_POST['project-uuid']) {
            delete_post_meta($post_id, 'project-uuid');
        }
        update_post_meta($post_id, 'project-uuid', $_POST['project-uuid']);
    }

    static function updateByUuid()
    {
        global $wpdb;
        $postTable = $wpdb->prefix . 'posts';
        $postMetaPost = $wpdb->prefix . 'postmeta';
        $uuids = $wpdb->get_results("
        SELECT pm.`meta_value`, p.ID FROM $postMetaPost as pm
        LEFT JOIN $postTable as p ON p.ID = pm.post_id
        WHERE p.post_type = 'prosjekt' AND pm.meta_key = 'project-uuid'", ARRAY_A);

        foreach ($uuids as $uuid){
            $uuidNew = $uuid['meta_value'];
            $postId = $uuid['ID'];
            $getResponse = ApiFunc::instance()->getProjectWithResources($uuidNew);
            if (!empty($getResponse)) {
                update_post_meta($postId, 'st_project_data', $getResponse);
                self::setFeaturedImage($postId, $getResponse);
            }
        }
        wp_send_json(['status' => 'success']);
    }

    static function fetchAllProjects()
    {
        $allProjects = ApiFunc::instance()->getAllProjects();

        foreach ($allProjects as $project) {
            self::insertProject($project, $project->uuid);
        }

        wp_send_json(['status' => 'success']);
    }

    static function setFeaturedImage($postId, $response) {
        $updateFeaturedImage = !!get_option('m360-sync-set-featured-image');

        if (!$updateFeaturedImage || !isset($response->images) || empty($response->images)) {
            return;
        }

        $apiImage = $response->images[0];
        $fileUrl = $apiImage->url;
        $fileName = $apiImage->filename;

        // Get current featured image
        $featuredImageId = get_post_thumbnail_id($postId);

        // Get current featured image filename
        $featuredImage = wp_get_attachment_image_src($featuredImageId, 'full');
        $featuredImageFilename = basename($featuredImage[0]);

        // If current featured image is the same as the one from the API, do nothing
        if ($featuredImageFilename === $fileName) {
            return;
        }

        // Replace the featured image
        $uploadDir = wp_upload_dir();
        $uploadPath = $uploadDir['path'];
        $uploadUrl = $uploadDir['url'];

        $fileData = file_get_contents($fileUrl);
        $filePath = $uploadPath . '/m360-images/' . $fileName;

        if (!file_exists($uploadPath . '/m360-images')) {
            mkdir($uploadPath . '/m360-images');
        }

        file_put_contents($filePath, $fileData);

        $wpFiletype = wp_check_filetype($fileName, null);
        $attachment = array(
            'post_mime_type' => $wpFiletype['type'],
            'post_title' => sanitize_file_name($fileName),
            'post_content' => '',
            'post_status' => 'inherit'
        );
        $attachId = wp_insert_attachment($attachment, $filePath, $postId);
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $attachData = wp_generate_attachment_metadata($attachId, $filePath);
        wp_update_attachment_metadata($attachId, $attachData);
        set_post_thumbnail($postId, $attachId);
    }
}
